'
' SAP TRP/DPI Document Printing Automation Script
' Author: Dario Pascoal
'
' Description: This VBScript automates the process of locating, opening, and 
' printing TRP (PICT) and DPI technical documentation files from the SAP ZPACK 
' transaction system. The script handles multiple document viewers (PowerPoint, 
' Adobe Reader/Acrobat, Microsoft Edge) and provides intelligent printing 
' automation with configurable copy quantities for DHL technical documentation workflows.
'
' Business Purpose:
' In DHL's technical documentation process, TRP (Technical Reference Procedures) 
' and DPI (Digital Process Instructions) documents need to be printed for field 
' operations and quality control procedures. This script automates the entire 
' workflow from SAP document retrieval to physical printing, eliminating manual 
' steps and ensuring consistent document output quality.
'
' Workflow Integration:
' 1. Receives material number and print quantity from calling application
' 2. Connects to active SAP GUI session for document access
' 3. Navigates to ZPACK transaction for material documentation lookup
' 4. Searches for specified material number and locates documentation files
' 5. Prioritizes TRP files over DPI files based on business requirements
' 6. Opens documents in appropriate viewer application automatically
' 7. Detects running application type and applies specific printing procedures
' 8. Configures print quantity and executes printing operation
' 9. Closes document viewers and returns to clean SAP state
'
' Technical Implementation:
' - VBScript with COM object automation for SAP GUI interaction
' - Multi-application detection and handling (PowerPoint, Adobe, Edge)
' - Process monitoring and window management for document viewers
' - Intelligent keyboard automation for printing dialog navigation
' - Command-line interface with material number and quantity parameters
' - Comprehensive error handling with graceful degradation
' - File-based logging system for audit trails and debugging
' - Registry security configuration for unattended automation
'
' Integration Points:
' - SAP GUI Scripting Engine for automated navigation and document access
' - Microsoft PowerPoint for .pptx/.ppt document handling and printing
' - Adobe Reader/Acrobat for PDF document viewing and printing
' - Microsoft Edge for alternative PDF viewing and web-based printing
' - Windows Shell for process detection and keyboard automation
' - Windows Task Manager integration for reliable application closure
' - DHL All-in-One main application for status reporting and workflow control
'
' Security Features:
' - Temporary registry modifications with automatic cleanup procedures
' - Error handling to prevent script crashes and document corruption
' - Detailed logging for security audit trails and compliance tracking
' - Secure SAP GUI connection handling with proper authentication
' - Automatic cleanup procedures to prevent resource leaks and open documents
' - Process termination safeguards to ensure clean application closure
'
' Document Type Priorities:
' 1. TRP (PICT) Files: Primary technical reference documents (highest priority)
' 2. DPI Files: Digital process instructions (secondary priority when TRP unavailable)
'
' Application Handling Logic:
' 1. PowerPoint: Native printing with tab navigation and quantity configuration
' 2. Adobe Reader/Acrobat: PDF printing with Alt+C shortcut and quantity setup
' 3. Microsoft Edge: Web-based PDF printing with tabbed navigation
'
' Return Codes:
' - "SUCCESS: [filename]": TRP document successfully printed
' - "DPI-SUCCESS": DPI document successfully printed when TRP unavailable
' - "NOT-FOUND": No suitable documentation files found for material
'

' =============================================================================
' GLOBAL ERROR HANDLING CONFIGURATION
' =============================================================================
' Enable comprehensive error handling to prevent script crashes and provide
' graceful degradation when encountering unexpected conditions or application issues
On Error Resume Next

' =============================================================================
' ENVIRONMENT INITIALIZATION
' =============================================================================
' Initialize Windows Script Host shell object for system interactions
' This object handles registry modifications and security dialog automation

' Create Windows Script Host shell object for registry and system operations
' This enables registry modifications needed for SAP GUI automation
Set WSHShell = CreateObject("WScript.Shell")

' =============================================================================
' SAP GUI SECURITY CONFIGURATION
' =============================================================================
' Disable SAP GUI scripting security warnings to enable automated interaction
' These registry modifications are essential for unattended script execution
' and prevent user prompts that would interrupt the automation workflow

' Disable attachment warnings when scripts connect to SAP GUI sessions
' This prevents popup dialogs that would halt automation processes
WSHShell.RegWrite "HKEY_CURRENT_USER\Software\SAP\SAPGUI Front\SAP Frontend Server\Security\WarnOnAttach", 0, "REG_DWORD"

' Disable connection warnings for script-to-SAP communication
' This ensures seamless connection establishment without user intervention
WSHShell.RegWrite "HKEY_CURRENT_USER\Software\SAP\SAPGUI Front\SAP Frontend Server\Security\WarnOnConnection", 0, "REG_DWORD"

' Disable allow list requirement warnings for automated scripting
' This prevents security prompts that would require manual user interaction
WSHShell.RegWrite "HKEY_CURRENT_USER\Software\SAP\SAPGUI Front\SAP Frontend Server\Security\WarnOnAllowListRequired", 0, "REG_DWORD"

' Disable allow list usage to prevent script blocking
' This allows our automation script to execute without whitelist restrictions
WSHShell.RegWrite "HKEY_CURRENT_USER\Software\SAP\SAPGUI Front\SAP Frontend Server\Security\UseAllowList", 0, "REG_DWORD"

' =============================================================================
' LOGGING SYSTEM INITIALIZATION
' =============================================================================
' Setup comprehensive logging system for operation tracking and debugging
' The logging system supports detailed file logging with timestamps for
' audit trails, troubleshooting, and operational monitoring

' Declare log file objects and path variables
' These objects handle all file-based logging operations
Dim logFile, logFilePath

' Initialize logging system if log file path is provided as third argument
' Enhanced logging provides detailed operation tracking for debugging and audit
If WScript.Arguments.Count > 2 Then
    logFilePath = WScript.Arguments.Item(2)
    
    ' Attempt to open log file in append mode for continuous logging
    ' Append mode (8) preserves existing log entries and adds new ones
    On Error Resume Next
    Set fso = CreateObject("Scripting.FileSystemObject")
    Set logFile = fso.OpenTextFile(logFilePath, 8, True)
    
    ' Handle log file opening errors gracefully
    ' If logging fails, continue script execution without logging
    If Err.Number <> 0 Then
        WScript.Echo "ERROR: Could not open log file: " & Err.Description
        Err.Clear
        Set logFile = Nothing
    Else
        ' Log script startup information for audit trail
        ' This creates a clear session marker in the log file
        LogToFile "========================================="
        LogToFile "Script Started at " & Now
        LogToFile "Script: " & WScript.ScriptName
        LogToFile "Material Number: " & materialNumber
        LogToFile "Quantity: " & quantity
        LogToFile "========================================="
    End If
End If

' =============================================================================
' LOGGING UTILITY FUNCTIONS
' =============================================================================
' Centralized logging functions to ensure consistent operation tracking
' and error reporting across all script operations and application interactions

'
' File-only logging function for detailed internal operations
' Used for technical details that don't need console output
'
Sub LogToFile(message)
    On Error Resume Next
    ' Only write to file if logging is enabled and file is available
    If Not logFile Is Nothing Then
        ' Format: timestamp - [script name] - message for easy parsing
        logFile.WriteLine Now & " - [" & WScript.ScriptName & "] - " & message
    End If
End Sub

'
' Dual logging function for important messages and user feedback
' Outputs to both console (for immediate feedback) and log file (for records)
'
Sub LogMessage(message)
    ' Display message to user for immediate feedback
    WScript.Echo message
    ' Also save to log file for permanent record
    LogToFile message
End Sub

' =============================================================================
' COMMAND LINE ARGUMENT PROCESSING
' =============================================================================
' Extract and validate command line arguments for material number and print quantity
' The script requires a material number and optionally accepts print quantity and log path

' Get material number from first command line argument (required)
' This material number identifies the equipment for TRP/DPI document printing
materialNumber = WScript.Arguments.Item(0)

' Get print quantity from second command line argument (optional, defaults to 1)
' This determines how many copies of the document will be printed
quantity = 1  ' Default to 1 copy for standard operations
If WScript.Arguments.Count > 1 Then
    ' Convert string argument to integer for print quantity processing
    quantity = CInt(WScript.Arguments.Item(1))
End If

' =============================================================================
' SAP GUI CONNECTION ESTABLISHMENT
' =============================================================================
' Establish connection to SAP GUI application and active session
' This section handles the complex process of connecting to SAP through
' the SAP GUI Scripting Engine and manages connection error handling

' Initialize SAP GUI application object if not already connected
' The application object is the top-level SAP GUI automation interface
If Not IsObject(application) Then
   ' Connect to the SAP GUI Automation Server
   ' This server manages all SAP GUI automation and scripting operations
   Set SapGuiAuto = GetObject("SAPGUI")
   ' Get the scripting engine from the SAP GUI server
   ' The scripting engine provides programmatic access to SAP GUI elements
   Set application = SapGuiAuto.GetScriptingEngine
End If

' Initialize SAP connection object if not already established
' The connection object represents a specific SAP system connection
If Not IsObject(connection) Then
   ' Get the first (primary) connection from available connections
   ' Children(0) accesses the first active connection to SAP
   Set connection = application.Children(0)
End If

' Initialize SAP session object for user interaction simulation
' The session object provides access to SAP screens and transactions
If Not IsObject(session) Then
   ' Get the first (primary) session from the active connection
   ' This session will be used for all SAP GUI automation commands
   Set session = connection.Children(0)
End If

' Connect VBScript event handlers to SAP GUI objects
' This enables the script to respond to SAP GUI events and errors
If IsObject(WScript) Then
   ' Connect session events for user interaction monitoring
   ' "on" prefix enables event handling for session-level events
   WScript.ConnectObject session, "on"
   ' Connect application events for system-level monitoring
   ' This handles application-wide events and error conditions
   WScript.ConnectObject application, "on"
End If

' =============================================================================
' SECURITY DIALOG AUTOMATION
' =============================================================================
' Automated handling of SAP GUI security dialogs that may appear during
' script execution, ensuring unattended operation without user intervention

'
' Handle security dialog automation for unattended script execution
' This function automatically dismisses security warnings that could
' interrupt the automation workflow and require manual intervention
'
Sub HandleSecurityDialog()
    Dim shell, timeout, dialogFound
    Set shell = CreateObject("WScript.Shell")
    timeout = 0
    dialogFound = False
    
    ' Wait up to 10 seconds for security dialog to appear
    ' This prevents infinite waiting while allowing time for dialog display
    Do While timeout < 10
        ' Check if SAP GUI Security dialog is active
        ' AppActivate returns True if the specified window title is found
        If shell.AppActivate("SAP GUI Security") Then
            ' Brief pause to ensure dialog is fully loaded
            ' This prevents sending keys before dialog is ready
            WScript.Sleep 500
            
            ' Press space to check "Remember my decision" checkbox
            ' This prevents the same security dialog from appearing repeatedly
            shell.SendKeys " "
            
            ' Brief pause between key operations for reliability
            ' Ensures the checkbox state change is processed
            WScript.Sleep 200
            
            ' Press Enter to click the "Allow" button
            ' This grants permission for the script to continue execution
            shell.SendKeys "{ENTER}"
            dialogFound = True
            ' Exit loop immediately after successful dialog handling
            Exit Do
        End If
        
        ' Wait 500ms before checking again to avoid excessive CPU usage
        ' This provides a reasonable balance between responsiveness and performance
        WScript.Sleep 500
        timeout = timeout + 1
    Loop
End Sub

' =============================================================================
' APPLICATION DETECTION FUNCTIONS
' =============================================================================
' Functions to detect running document viewer applications for intelligent
' document handling and printing automation based on application type

'
' Function to detect if Microsoft PowerPoint is currently running
' This function checks for PowerPoint processes to determine document type handling
'
' Returns: Boolean - True if PowerPoint is detected, False otherwise
'
Function IsPowerPointRunning()
    Dim WshShell, oExec, line, isRunning
    Set WshShell = CreateObject("WScript.Shell")
    isRunning = False
    
    ' Execute tasklist command to check for PowerPoint processes
    ' This provides reliable process detection across Windows versions
    On Error Resume Next
    Set oExec = WshShell.Exec("tasklist /FI ""IMAGENAME eq POWERPNT.EXE""")
    
    ' Read command output to check for PowerPoint process presence
    ' Process detection is case-insensitive for better compatibility
    Do While Not oExec.StdOut.AtEndOfStream
        line = oExec.StdOut.ReadLine()
        ' Check if PowerPoint executable is listed in running processes
        If InStr(1, line, "POWERPNT.EXE", vbTextCompare) > 0 Then
            isRunning = True
            Exit Do
        End If
    Loop
    
    IsPowerPointRunning = isRunning
End Function

'
' Function to detect if Adobe Reader or Acrobat is currently running
' This function checks for various Adobe PDF viewer processes to determine
' PDF document handling requirements and printing procedures
'
' Returns: Boolean - True if any Adobe PDF viewer is detected, False otherwise
'
Function IsAdobeRunning()
    Dim WshShell, adobeProcesses, isRunning
    Set WshShell = CreateObject("WScript.Shell")
    isRunning = False
    
    ' Define array of possible Adobe process names for comprehensive detection
    ' This covers different Adobe versions and installation types
    adobeProcesses = Array("AcroRd32.exe", "Acrobat.exe", "AcroRd.exe", "AdobeAcrobatDC.exe", "reader_sl.exe")
    
    ' Check each Adobe process individually for reliable detection
    ' This ensures compatibility with various Adobe installations
    On Error Resume Next
    For Each process In adobeProcesses
        Dim cmd, oExec, output
        ' Build command to check for specific Adobe process
        cmd = "tasklist /FI ""IMAGENAME eq " & process & """ /NH"
        Set oExec = WshShell.Exec(cmd)
        output = oExec.StdOut.ReadAll()
        
        ' If process is found in output, Adobe is running
        ' Log detection for debugging and audit purposes
        If InStr(output, process) > 0 Then
            LogMessage "DEBUG: Found Adobe process: " & process
            isRunning = True
            Exit For
        End If
    Next
    
    IsAdobeRunning = isRunning
End Function

'
' Function to detect if Microsoft Edge is currently running
' This function checks for Edge processes to determine web-based PDF handling
' requirements for documents that open in Edge's built-in PDF viewer
'
' Returns: Boolean - True if Microsoft Edge is detected, False otherwise
'
Function IsEdgeRunning()
    Dim WshShell, oExec, line, isRunning
    Set WshShell = CreateObject("WScript.Shell")
    isRunning = False
    
    ' Check for Microsoft Edge processes using tasklist command
    ' Edge may handle PDF documents through its built-in viewer
    On Error Resume Next
    Set oExec = WshShell.Exec("tasklist /FI ""IMAGENAME eq msedge.exe"" /NH")
    
    ' Read command output to detect Edge process presence
    If Not oExec.StdOut.AtEndOfStream Then
        line = oExec.StdOut.ReadAll()
        ' Check if Edge executable is found in process list
        If InStr(line, "msedge.exe") > 0 Then
            isRunning = True
            LogMessage "DEBUG: Microsoft Edge detected"
        End If
    End If
    
    IsEdgeRunning = isRunning
End Function

' =============================================================================
' APPLICATION-SPECIFIC PRINTING HANDLERS
' =============================================================================
' Specialized functions for handling document printing in different applications
' Each function implements application-specific printing procedures and automation

'
' Enhanced PowerPoint document printing handler with precise navigation
' This function handles PowerPoint presentations with single-tab navigation
' to the copies field for accurate quantity configuration
'
' Parameters:
' - shell: WScript.Shell object for keyboard automation
' - quantity: Number of copies to print (integer)
'
Sub HandlePowerPoint(shell, quantity)
    ' Clear any pending keyboard input to ensure clean operation
    ' This prevents interference from previous keyboard operations
    Err.Clear
    
    ' Send Ctrl+P to open PowerPoint's print dialog
    ' This is the standard Windows shortcut for printing in Office applications
    shell.SendKeys "^p"
    WScript.Sleep 2000  ' Wait for print dialog to fully load and display
    
    ' Navigate to copies field using single tab operation
    ' PowerPoint's print dialog has a specific tab order for field navigation
    LogMessage "DEBUG: Sending single tab to PowerPoint dialog"
    shell.SendKeys "{TAB}"
    LogMessage "DEBUG: Tab sent"
    WScript.Sleep 1500  ' Extended wait to ensure focus is properly set on copies field
    
    ' Clear existing quantity value and set new quantity
    ' HOME+END+DEL ensures complete field clearing regardless of current content
    shell.SendKeys "{HOME}+{END}{DEL}"
    WScript.Sleep 800   ' Brief pause to ensure field is cleared
    shell.SendKeys quantity
    WScript.Sleep 1500  ' Extended wait to ensure quantity value is properly entered
    
    ' Execute print operation using Enter key
    ' Enter key confirms the print operation with current settings
    shell.SendKeys "{ENTER}"
    WScript.Sleep 5000  ' Wait for print job to be submitted and processing to begin
    
    ' Force close PowerPoint using taskkill for reliable cleanup
    ' This ensures complete application closure and prevents document conflicts
    On Error Resume Next
    Set oShell = CreateObject("WScript.Shell")
    oShell.Run "taskkill /IM POWERPNT.EXE /F", 0, True
    On Error GoTo 0
    
    WScript.Sleep 1500  ' Allow additional time for complete application termination
End Sub

'
' Enhanced PDF document printing handler for Adobe Reader/Acrobat
' This function handles PDF documents with direct window activation and
' Alt+C shortcut navigation for reliable copies field access
'
' Parameters:
' - shell: WScript.Shell object for keyboard automation
' - quantity: Number of copies to print (integer)
' - docType: Document type identifier for logging purposes
'
Sub HandlePdf(shell, quantity, docType)
    ' Clear any pending keyboard input to ensure clean operation
    ' This prevents interference from previous automation operations
    Err.Clear
    
    ' Allow sufficient time for PDF document to fully load in Adobe viewer
    ' PDF documents may require additional loading time for complex content
    LogMessage "BASIC-DEBUG: Waiting for PDF to fully load..."
    WScript.Sleep 3000
    
    ' ==========================================================================
    ' WINDOW ACTIVATION - DIRECT METHODS ONLY
    ' ==========================================================================
    ' Attempt to activate Adobe Reader window using multiple title variations
    ' This section uses only reliable direct window activation methods
    
    LogMessage "BASIC-DEBUG: Attempting to activate Adobe Reader window..."
    Dim activated : activated = False
    
    ' Try multiple Adobe window title variations for maximum compatibility
    ' Different Adobe versions and configurations may use different window titles
    If shell.AppActivate("Adobe Acrobat Reader") Then
        activated = True
        LogMessage "BASIC-DEBUG: Activated Adobe Acrobat Reader directly"
    ElseIf shell.AppActivate("Adobe Acrobat") Then
        activated = True
        LogMessage "BASIC-DEBUG: Activated Adobe Acrobat directly"
    ElseIf shell.AppActivate("Acrobat") Then
        activated = True
        LogMessage "BASIC-DEBUG: Activated Acrobat directly"
    ElseIf shell.AppActivate(".pdf") Then
        activated = True
        LogMessage "BASIC-DEBUG: Activated window with .pdf in title"
    End If
    
    ' Continue with printing even if window activation fails
    ' The keyboard commands may still reach the correct application
    If Not activated Then
        LogMessage "BASIC-DEBUG: Could not activate window directly, will try to send keys anyway"
    End If
    
    ' Additional delay before sending print commands for stability
    WScript.Sleep 1000
    
    ' ==========================================================================
    ' PDF PRINTING AUTOMATION
    ' ==========================================================================
    ' Execute PDF printing workflow using Adobe-specific keyboard shortcuts
    
    ' Send Ctrl+P to open Adobe's print dialog
    ' This is the universal printing shortcut for Adobe applications
    LogMessage "BASIC-DEBUG: Sending Ctrl+P to open print dialog"
    shell.SendKeys "^p"
    WScript.Sleep 2000  ' Allow sufficient time for print dialog to open and render
    
    ' Use Alt+C shortcut to navigate directly to copies field
    ' This is Adobe's standard keyboard shortcut for the copies field
    LogMessage "BASIC-DEBUG: Sending Alt+C for copies field"
    shell.SendKeys "%c"
    WScript.Sleep 1000  ' Wait for focus to move to copies field
    
    ' Clear existing value and enter new quantity
    ' This ensures accurate quantity setting regardless of default values
    LogMessage "BASIC-DEBUG: Setting quantity to " & quantity
    shell.SendKeys "{HOME}+{END}{DEL}"
    WScript.Sleep 800   ' Allow time for field clearing
    shell.SendKeys quantity
    WScript.Sleep 1500  ' Ensure quantity value is properly entered
    
    ' Execute print operation using Enter key
    ' Enter confirms the print with current dialog settings
    LogMessage "BASIC-DEBUG: Sending Enter to print"
    shell.SendKeys "{ENTER}"
    WScript.Sleep 4000  ' Wait for print job submission and initial processing
    
    ' Force close Adobe applications using taskkill for complete cleanup
    ' This ensures no documents remain open after printing completion
    LogMessage "BASIC-DEBUG: Force closing Adobe"
    On Error Resume Next
    Set oShell = CreateObject("WScript.Shell")
    oShell.Run "taskkill /IM AcroRd32.exe /F", 0, True     ' Adobe Reader
    oShell.Run "taskkill /IM Acrobat.exe /F", 0, True      ' Adobe Acrobat
    On Error GoTo 0
    
    WScript.Sleep 1500  ' Allow time for complete application termination
    LogMessage "BASIC-DEBUG: Adobe PDF handling completed for " & docType
End Sub

'
' Enhanced Microsoft Edge PDF printing handler for web-based PDF viewing
' This function handles PDFs that open in Edge's built-in PDF viewer with
' specialized navigation for web-based print dialogs
'
' Parameters:
' - shell: WScript.Shell object for keyboard automation  
' - quantity: Number of copies to print (integer)
'
Sub HandleEdgePdf(shell, quantity)
    ' Clear any pending keyboard input to ensure clean operation
    ' This prevents interference from previous automation operations
    Err.Clear
    
    ' Allow extended time for Edge to fully load PDF content
    ' Web-based PDF viewers require additional loading time
    WScript.Sleep 5000
    LogMessage "DEBUG: Starting Edge PDF print procedure"
    
    ' ==========================================================================
    ' EDGE WINDOW ACTIVATION
    ' ==========================================================================
    ' Attempt to activate Microsoft Edge window using multiple title variations
    ' Edge may use different window titles depending on PDF content and version
    
    Dim activated : activated = False
    
    ' Try different Edge window title patterns for maximum compatibility
    ' Edge window titles may vary based on PDF filename and browser state
    If shell.AppActivate("Microsoft Edge") Then
        activated = True
        LogMessage "DEBUG: Activated 'Microsoft Edge'"
    ElseIf shell.AppActivate(".pdf - Microsoft Edge") Then
        activated = True
        LogMessage "DEBUG: Activated '.pdf - Microsoft Edge'"
    ElseIf shell.AppActivate("PDF Viewer - Microsoft Edge") Then
        activated = True
        LogMessage "DEBUG: Activated 'PDF Viewer - Microsoft Edge'"
    End If
    
    ' If direct activation fails, try generic Edge activation
    ' This provides fallback activation for non-standard window titles
    If Not activated Then
        LogMessage "DEBUG: Unable to activate Edge window directly, trying broader approach"
        shell.AppActivate("Edge")
    End If
    
    ' ==========================================================================
    ' EDGE PDF PRINTING AUTOMATION
    ' ==========================================================================
    ' Execute PDF printing workflow using Edge's web-based print interface
    
    ' Send Ctrl+P to open Edge's print dialog
    ' Edge uses web-based print interface with different navigation patterns
    LogMessage "DEBUG: Sending Ctrl+P to open print dialog in Edge"
    shell.SendKeys "^p"
    WScript.Sleep 3000  ' Extended wait for Edge's web-based print dialog to load
    
    ' Navigate to copies field using tabbed navigation
    ' Edge print dialog requires multiple tabs to reach the copies field
    LogMessage "DEBUG: Navigating to copies field in Edge print dialog"
    For i = 1 To 3  ' Edge typically requires 3 tabs to reach copies field
        shell.SendKeys "{TAB}"
        WScript.Sleep 600  ' Pause between tabs for reliable navigation
    Next
    
    ' Set print quantity directly in the copies field
    ' Edge may not require field clearing due to web-based interface behavior
    LogMessage "DEBUG: Setting quantity to " & quantity & " in Edge print dialog"
    shell.SendKeys quantity
    WScript.Sleep 1500  ' Ensure quantity value is properly entered
    
    ' Execute print operation using Enter key
    ' Enter confirms the print operation in Edge's web-based interface
    LogMessage "DEBUG: Sending Enter to print from Edge"
    shell.SendKeys "{ENTER}"
    WScript.Sleep 5000  ' Wait for print job submission and initial processing
    
    LogMessage "DEBUG: Finished Edge PDF print procedure, Edge will stay open"
    ' Note: Edge browser is intentionally left open for continued user access
    ' This differs from PowerPoint and Adobe handlers that close applications
End Sub

' =============================================================================
' DOCUMENT PROCESSING ORCHESTRATION
' =============================================================================
' Central document processing function that coordinates application detection,
' printing workflow execution, and success/failure logging for all document types

'
' Master document processing function that orchestrates the complete printing workflow
' This function handles application detection, prioritized printing logic, and
' comprehensive logging for both TRP and DPI document types
'
' Parameters:
' - fileName: Name of the document file being processed (string)
' - docType: Type of document ("TRP" or "DPI") for workflow customization
'
Sub ProcessDocument(fileName, docType)
    ' Wait for the document viewer application to fully load
    ' This delay ensures the document is completely rendered before automation begins
    WScript.Sleep 3000
    
    ' Create shell object for keyboard automation and application control
    ' The shell object provides system-level automation capabilities
    Set shell = CreateObject("WScript.Shell")
    
    ' ==========================================================================
    ' APPLICATION DETECTION AND PRIORITIZATION
    ' ==========================================================================
    ' Determine which document viewer application is running and prioritize
    ' handling based on application capabilities and reliability
    
    ' Perform comprehensive application detection for intelligent workflow routing
    ' This detection determines which printing handler to use for optimal results
    LogMessage "BASIC-DEBUG: Starting process document checks"
    Dim isPowerPoint, isAdobe, isEdge
    isPowerPoint = IsPowerPointRunning()
    isAdobe = IsAdobeRunning()
    isEdge = IsEdgeRunning()
    
    ' Log detection results for troubleshooting and audit trail
    ' This provides visibility into application detection logic for debugging
    LogMessage "BASIC-DEBUG: PowerPoint detected: " & isPowerPoint
    LogMessage "BASIC-DEBUG: Adobe detected: " & isAdobe
    LogMessage "BASIC-DEBUG: Microsoft Edge detected: " & isEdge
    
    ' ==========================================================================
    ' PRIORITIZED APPLICATION HANDLING
    ' ==========================================================================
    ' Execute printing workflow using prioritized application handling based on
    ' reliability, feature support, and successful printing capabilities
    
    ' Priority 1: Microsoft PowerPoint (most reliable for presentation documents)
    ' PowerPoint provides excellent automation support and reliable printing
    If isPowerPoint Then
        LogMessage "BASIC-DEBUG: Using PowerPoint handler"
        HandlePowerPoint shell, quantity
        LogMessage "BASIC-DEBUG: PowerPoint document printed and closed"
    
    ' Priority 2: Adobe PDF Reader/Acrobat (specialized PDF handling)
    ' Adobe applications provide comprehensive PDF printing with advanced options
    ElseIf isAdobe Then
        LogMessage "BASIC-DEBUG: Using Adobe PDF handler"
        HandlePdf shell, quantity, docType
        LogMessage "BASIC-DEBUG: PDF document printed and closed"
    
    ' Priority 3: Microsoft Edge (web-based PDF viewer, limited automation)
    ' Edge is used as fallback when other applications are unavailable
    ElseIf isEdge Then
        LogMessage "BASIC-DEBUG: Using Edge PDF handler"
        HandleEdgePdf shell, quantity
        LogMessage "BASIC-DEBUG: Edge PDF document printed (browser left open)"
        
        ' Navigate back in SAP for Edge only (other apps auto-close)
        ' Edge browser remains open, requiring manual SAP navigation
        LogMessage "BASIC-DEBUG: Navigating back in SAP since Edge stays open"
        session.findById("wnd[0]/tbar[0]/btn[3]").press
        session.findById("wnd[0]/tbar[0]/btn[3]").press
        session.findById("wnd[0]/tbar[0]/btn[3]").press
    
    Else
        ' =======================================================================
        ' FALLBACK APPLICATION DETECTION
        ' =======================================================================
        ' No applications detected initially - implement extended detection with
        ' longer wait times to handle slower document loading scenarios
        
        LogMessage "BASIC-DEBUG: No known application detected. Waiting longer..."
        WScript.Sleep 5000  ' Extended wait for slow document loading
        
        ' Retry application detection after extended wait period
        ' This handles cases where documents take longer to load in viewer applications
        isPowerPoint = IsPowerPointRunning()
        isAdobe = IsAdobeRunning()
        isEdge = IsEdgeRunning()
        
        ' Execute delayed application handling with same priority sequence
        ' Maintain consistent priority: PowerPoint, Adobe, Edge for reliability
        If isPowerPoint Then
            HandlePowerPoint shell, quantity
            LogMessage "BASIC-DEBUG: PowerPoint document printed and closed (delayed)"
        
        ElseIf isAdobe Then
            HandlePdf shell, quantity, docType
            LogMessage "BASIC-DEBUG: PDF document printed and closed (delayed)"
        
        ElseIf isEdge Then
            HandleEdgePdf shell, quantity
            LogMessage "BASIC-DEBUG: Edge PDF document printed (browser left open) (delayed)"
            
            ' Navigate back in SAP for delayed Edge processing
            LogMessage "BASIC-DEBUG: Navigating back in SAP since Edge stays open (delayed case)"
            session.findById("wnd[0]/tbar[0]/btn[3]").press
            session.findById("wnd[0]/tbar[0]/btn[3]").press
            session.findById("wnd[0]/tbar[0]/btn[3]").press
        Else
            ' No applications detected even after extended wait - log error
            LogMessage "BASIC-DEBUG: ERROR: No supported application detected for printing"
        End If
    End If
    
    ' ==========================================================================
    ' SUCCESS LOGGING AND WORKFLOW COMPLETION
    ' ==========================================================================
    ' Log successful processing completion with appropriate success indicators
    ' based on document type and file name for operational tracking
    
    ' Log success with document-specific information for audit trail
    ' This provides confirmation that the document was successfully processed
    If docType = "TRP" Then
        LogMessage "SUCCESS: " & fileName
    Else
        LogMessage "DPI-SUCCESS"
    End If
End Sub

' =============================================================================
' MAIN DOCUMENT PROCESSING WORKFLOW
' =============================================================================
' Core document processing workflow that manages SAP navigation, document
' discovery, prioritization, and automated printing through the ZPACK transaction

' Call HandleSecurityDialog before starting operations
HandleSecurityDialog()

' =============================================================================
' SAP TRANSACTION EXECUTION
' =============================================================================
' Navigate to ZPACK transaction and execute material search to locate
' TRP and DPI documents associated with the specified material number

' Maximize SAP window for optimal visibility and element access
session.findById("wnd[0]").maximize

' Navigate to ZPACK transaction for material document management
' ZPACK provides access to material-related documents including TRP and DPI files
session.findById("wnd[0]/tbar[0]/okcd").text = "zpack"
session.findById("wnd[0]").sendVKey 0

' =============================================================================
' MATERIAL SEARCH AND DOCUMENT DISCOVERY
' =============================================================================
' Execute material number search to populate document list and prepare
' for document type filtering and priority-based selection

' Enter material number in search field and execute search
' This populates the document table with all available documents for the material
session.findById("wnd[0]/usr/ctxtS_MATNR-LOW").text = materialNumber
session.findById("wnd[0]").sendVKey 8

' Get reference to the file list table for document enumeration
' This table contains all documents associated with the material number
Set fileTable = session.findById("wnd[0]/usr/tabsTAB_9100/tabpTAB_9100_FC3/ssubTAB_9100_SCA:ZDGP_MATERIAL_OVERVIEW:9103/cntlCONTROL_9103_2/shellcont/shell")

' =============================================================================
' DOCUMENT TYPE IDENTIFICATION AND PRIORITIZATION
' =============================================================================
' Scan document table to identify TRP and DPI documents with priority-based
' selection logic favoring TRP over DPI documents

' Initialize document discovery variables
' These track found documents and their table positions for selection
found = False
foundDPI = False
rowIndex = -1
dpiRowIndex = -1
foundFileName = ""

' Loop through the files to find TRP and DPI files
For i = 0 To fileTable.RowCount - 1
    fileName = fileTable.GetCellValue(i, "DOKNR") ' Document number column
    
    ' Check for TRP (PICT) files (highest priority)
    ' TRP documents contain technical inspection procedures and have priority
    ' over DPI documents when both are available for the same material
    If InStr(1, fileName, "PICT", vbTextCompare) > 0 Then
        found = True
        rowIndex = i
        foundFileName = fileName
        Exit For
    ' Check for DPI files (secondary priority)
    ' DPI documents contain detailed part inspections and are used when TRP unavailable
    ElseIf InStr(1, fileName, "DPI", vbTextCompare) > 0 Then
        foundDPI = True
        dpiRowIndex = i
        ' Don't exit, continue looking for PICT which has priority
    End If
Next

' =============================================================================
' DOCUMENT SELECTION AND PROCESSING EXECUTION
' =============================================================================
' Execute document selection based on priority hierarchy and trigger
' appropriate document processing workflow for the selected document type

' Handle TRP file if found (highest priority)
If found Then
    ' TRP document found - select and process it
    ' TRP documents take precedence over DPI documents for inspection procedures
    fileTable.setCurrentCell rowIndex, "DOKNR" ' Select TRP document row
    fileTable.selectedRows = rowIndex
    fileTable.doubleClickCurrentCell
    
    ' Process the TRP document using specialized TRP workflow
    ProcessDocument foundFileName, "TRP"
    
' Handle DPI file if found and TRP not found (secondary priority)
ElseIf foundDPI Then
    ' No TRP found, but DPI available - process DPI document
    ' DPI documents provide alternative inspection procedures when TRP unavailable
    fileTable.setCurrentCell dpiRowIndex, "DOKNR" ' Select DPI document row
    fileTable.selectedRows = dpiRowIndex
    fileTable.doubleClickCurrentCell
    
    ' Process the DPI document using specialized DPI workflow
    ProcessDocument "DPI", "DPI"
Else
    ' No relevant documents found - log status for troubleshooting
    ' This indicates either no documents exist or document naming doesn't match patterns
    LogMessage "NOT-FOUND"
End If

' =============================================================================
' SAP NAVIGATION CLEANUP
' =============================================================================
' Navigate back through SAP transaction screens to return to starting state
' This cleanup ensures proper SAP session state for subsequent operations

' Navigate back in SAP using back button sequence
' Three back button presses return to the main ZPACK screen from document view
session.findById("wnd[0]/tbar[0]/btn[3]").press
session.findById("wnd[0]/tbar[0]/btn[3]").press
session.findById("wnd[0]/tbar[0]/btn[3]").press

' =============================================================================
' CLEANUP AND FINALIZATION PROCEDURES
' =============================================================================
' Comprehensive cleanup procedures to ensure proper resource management
' and logging finalization before script termination

'
' Cleanup function to properly close log file and release system resources
' This function ensures proper log file closure and resource cleanup to
' prevent memory leaks and ensure all log data is properly written to disk
'
Sub CloseLogFile()
    On Error Resume Next
    ' Check if log file is still open and available for closure
    If Not logFile Is Nothing Then
        ' Write session end marker to log for clear audit trail separation
        LogToFile "Script ended at " & Now
        LogToFile "========================================="
        
        ' Properly close the log file to ensure all buffered data is written
        logFile.Close
        
        ' Release the log file object to free system resources
        Set logFile = Nothing
    End If
End Sub

' Execute final cleanup procedures before script termination
' This ensures proper resource cleanup and log file closure in all cases
CloseLogFile()